<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/* -------------------------------------------------
 *  Register the widget
 * ------------------------------------------------- */
add_action( 'widgets_init', function() {
    register_widget( 'MSPW_Multi_Select_Widget' );
} );

/* -------------------------------------------------
 *  Enqueue admin scripts (only on Widgets screen)
 * ------------------------------------------------- */
add_action( 'admin_enqueue_scripts', function( $hook ) {
    if ( 'widgets.php' !== $hook ) {
        return;
    }

    wp_enqueue_style(
        'select2',
        'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css',
        [],
        '4.1.0'
    );

    wp_enqueue_script(
        'select2',
        'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js',
        [ 'jquery' ],
        '4.1.0',
        true
    );

    wp_enqueue_script(
        'mspw-admin',
        get_template_directory_uri(). '/assets/js/mspw.js',
        [ 'jquery', 'select2' ],
        '1.1',
        true
    );


    wp_add_inline_style('wp-admin', '
	.select2-container--default .select2-selection--multiple {
		background-color: unset!important;
	}
	.select2-dropdown.select2-dropdown--below, .select2-container--default .select2-selection--multiple .select2-selection__choice {
		background-color: white!important;
		color: black!important;
	}
	.wp-die-message, p { 
		padding-top: 3rem!important 0;
	}
	.select2-container { 
		margin-bottom: 3rem!important;
	}
    ');
	
    wp_localize_script( 'mspw-admin', 'mspw', [
        'ajax_url' => admin_url( 'admin-ajax.php' ),
        'nonce'    => wp_create_nonce( 'mspw_search_nonce' ),
    ] );
} );

/* -------------------------------------------------
 *  AJAX handler – search posts
 * ------------------------------------------------- */
add_action( 'wp_ajax_mspw_search_posts', function() {
    check_ajax_referer( 'mspw_search_nonce', 'nonce' );

    $term = sanitize_text_field( $_GET['q'] ?? '' );

    $args = [
        'post_type'      => 'post',
        'post_status'    => 'publish',
        'posts_per_page' => -1,
        'orderby'        => 'date',
        'order'          => 'DESC',
        's'              => $term,
    ];

    $posts = get_posts( $args );

    $results = [];
    foreach ( $posts as $p ) {
        $results[] = [
            'id'   => $p->ID,
            'text' => $p->post_title.' ',
        ];
    }

    wp_send_json( [ 'results' => $results ] );
} );

/* -------------------------------------------------
 *  The widget class
 * ------------------------------------------------- */
class MSPW_Multi_Select_Widget extends WP_Widget {

    public function __construct() {
        parent::__construct(
            'mspw_multi_select',
            __( '('.THEMES_NAMES.') Pinned Post by Multi-Select Posts (Search)', TEXT_DOMAIN ),
            [ 'description' => __( 'Display Selected Pinned Post From by Multi-Select Posts (Search)', TEXT_DOMAIN ) ]
        );
    }

    /* ---- BACKEND FORM ---- */
    public function form( $instance ) {
        $title          = $instance['title'] ?? __( 'Pinned Post', 'mspw' );		
		$in				= $instance['in'] ?? __( 'in', 'mspw' );
		$published		= $instance['published'] ?? __( 'Published', 'mspw' );
		$readmore		= $instance['readmore'] ?? __( 'Read more', 'mspw' );
		$show_desc		= (bool) $instance['show_desc']; 
		
        $selected_posts = $instance['selected_posts'] ?? [];
        if ( ! is_array( $selected_posts ) ) {
            $selected_posts = [];
        }
		
        ?>
		
		<p class="info-notice danger" style="text-align: center;font-weight: bold;"><?php _e('Pick multiple posts with live search.', TEXT_DOMAIN); ?> </p>
        <p>
            <label for="<?php echo $this->get_field_id( 'title' ); ?>">
                <?php _e( 'Title:', TEXT_DOMAIN ); ?>
            </label>
        </p>
        <p>
            <input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>">
        </p>

		<p>
			<label for="<?php echo $this->get_field_id( 'readmore' ); ?>"><b>Title Button </b></label>
		</p> 
		<p>
			<input class="widefat" type="text" id="<?php echo $this->get_field_id( 'readmore' ); ?>" name="<?php echo $this->get_field_name( 'readmore' ); ?>" value="<?php echo esc_attr( $readmore ); ?>" />
		</p>
		<hr />
		<p>
			<label for="<?php echo $this->get_field_id( 'in' ); ?>"><b>Title IN </b></label>
		</p> 
		<p>
			<input class="widefat" type="text" id="<?php echo $this->get_field_id( 'in' ); ?>" name="<?php echo $this->get_field_name( 'in' ); ?>" value="<?php echo esc_attr( $in ); ?>" />
		</p>
		<hr />
		<p>
			<label for="<?php echo $this->get_field_id( 'published' ); ?>"><b>Title Published </b></label>
		</p> 
		<p>
			<input class="widefat" type="text" id="<?php echo $this->get_field_id( 'published' ); ?>" name="<?php echo $this->get_field_name( 'published' ); ?>" value="<?php echo esc_attr( $published ); ?>" />
		</p>
		<hr />
        <p>
            <label for="<?php echo $this->get_field_id( 'selected_posts' ); ?>">
                <?php _e( 'Select Posts:', TEXT_DOMAIN ); ?>
            </label>
        </p>
        <p>
            <select id="<?php echo $this->get_field_id( 'selected_posts' ); ?>" name="<?php echo $this->get_field_name( 'selected_posts' ); ?>[]" class="mspw-select2 widefat" multiple="multiple">
                <?php
                // Pre-populate already saved IDs
                if ( ! empty( $selected_posts ) ) {
                    $saved = get_posts( [
                        'post__in'       => $selected_posts,
                        'post_type'      => 'post',
                        'post_status'    => 'publish',
                        'posts_per_page' => -1,
                        'orderby'        => 'post__in',
                    ] );
                    foreach ( $saved as $p ) {
                        echo '<option value="' . esc_attr( $p->ID ) . '" selected>' . esc_html( $p->post_title ) . '</option>';
                    }
                }
                ?>
            </select>
        </p>
		
		<p class="info-notice danger" style="text-align: center;font-weight: bold;font-size: 12px;"><?php _e('Start typing a title to search. <br>Hold Ctrl/Cmd to pick many.', TEXT_DOMAIN); ?> </p>
		
		<hr />
		<p>
			<input class="checkbox" type="checkbox" <?php checked( $show_desc ); ?> id="<?php echo esc_html( $this->get_field_id( 'show_desc' ) ); ?>" name="<?php echo esc_html( $this->get_field_name( 'show_desc' ) ); ?>" /><label for="<?php echo esc_html( $this->get_field_id( 'show_desc' ) ); ?>"><?php _e( 'Show Excerpt?', TEXT_DOMAIN ); ?></label>
		</p>
		<hr />
	
        <?php
    }

    /* ---- SAVE SETTINGS ---- */
    public function update( $new_instance, $old_instance ) {
        $instance                 	= [];
        $instance['title']        	= sanitize_text_field( $new_instance['title'] ?? '' );
		$instance['readmore']		= strip_tags( $new_instance[ 'readmore' ] ?? '' ); 
		$instance['in']				= strip_tags( $new_instance[ 'in' ] ?? '' ); 
		$instance['published']		= strip_tags( $new_instance[ 'published' ] ?? '' ); 
		$instance['show_desc']		= (bool) $new_instance['show_desc'];
		
        $instance['selected_posts'] = ! empty( $new_instance['selected_posts'] )
            ? wp_parse_id_list( $new_instance['selected_posts'] )
            : [];

		
        return $instance;
    }

    /* ---- FRONTEND DISPLAY ---- */
    public function widget( $args, $instance ) {
        $widget_id			= $this->id_base . '-' . $this->number;
        //echo $args['before_widget'];

        //if ( ! empty( $instance['title'] ) ) {
        //    echo $args['before_title'] . apply_filters( 'widget_title', $instance['title'] ) . $args['after_title'];
        //}

        $selected = $instance['selected_posts'] ?? [];

        //if ( empty( $selected ) ) {
        //    echo '<p>' . __( 'No posts selected.', TEXT_DOMAIN ) . '</p>';
        //    echo $args['after_widget'];
        //    return;
        //}

        $query = new WP_Query( [
            'post__in'       => $selected,
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'orderby'        => 'post__in',   // keep the order you chose
        ] );

        if ( $query->have_posts() ) {
		//echo '<ul class="mspw-post-list">';
		set_query_var( 'in', $instance['in'] ); 
		set_query_var( 'published', $instance['published'] ); 
		set_query_var( 'readmore', $instance['readmore'] ); 
		set_query_var( 'show_desc', $instance['show_desc'] ); 
		?>


	<div class='widget FeaturedPost' id="<?php echo $widget_id; ?>">
	<h2 class='title dt'><?php _e($instance['title'], TEXT_DOMAIN); ?></h2>
	<div class='itemFt' role='feed'>
	 
		<?php
		while ( $query->have_posts() ) {
		$query->the_post();                
		get_template_part('loop/item.featured');
		}
		?>

	</div>
	</div>
		<?php
		//echo '</ul>';
		wp_reset_postdata();
        }

        //echo $args['after_widget'];
    }
}

/* -------------------------------------------------
 *  Admin JS – initialise Select2 with AJAX
 * ------------------------------------------------- */
add_action( 'admin_footer', function() {
    // Only output on widgets page
    $screen = get_current_screen();
    if ( ! $screen || 'widgets' !== $screen->id ) {
        return;
    }
    ?>
    <script type="text/javascript">
        jQuery(document).ready(function($){
            $('.mspw-select2').each(function(){
                var $select = $(this);
                $select.select2({
                    placeholder: "<?php _e( 'Search posts…', 'mspw' ); ?>",
                    allowClear: true,
                    minimumInputLength: 2,
                    ajax: {
                        url: mspw.ajax_url,
                        dataType: 'json',
                        delay: 250,
                        data: function(params){
                            return {
                                q: params.term,
                                action: 'mspw_search_posts',
                                nonce: mspw.nonce
                            };
                        },
                        processResults: function(data){
                            return data;
                        },
                        cache: true
                    }
                });
            });
        });
    </script>
    <?php
}, 999 );